/* Logo Spin Text — league-only PNG sequence per team
   Project: ScoringAnimation1.aep
   Comp:    Logo Spin Text
*/

(function () {
  // ---- Load common helpers (force reload) ----
  (function(){
    var COMMON = $.getenv("AE_COMMON_JSX") || "";
    if (COMMON) { try{ delete $.global.GL; }catch(e){ $.global.GL = undefined; } $.evalFile(File(COMMON)); }
    if (!$.global.GL) throw new Error("Common library not loaded. Set AE_COMMON_JSX to gl_common.jsxinc");
  })();

  var GL = $.global.GL;

  // ---- Env ----
  var PROJECT   = GL.env("AE_PROJECT", null);
  var CSV_PATH  = GL.env("AE_CSV", null);
  var COMP_NAME = GL.env("AE_COMP","Logo Spin Text");

  // Layer names (override via env if needed)
  var LYR_TEAMNAME       = GL.env("AE_TEAMNAME_LAYER",      "TeamName");
  var LYR_FIELDGOAL      = GL.env("AE_FIELDGOAL_LAYER",     "FIELD_GOAL"); // alt tried below
  var LYR_ROTATELOGO1    = GL.env("AE_ROTATELOGO1_LAYER",   "RotateLogo1");
  var LYR_ROTATELOGO2    = GL.env("AE_ROTATELOGO2_LAYER",   "RotateLogo2");
  var LYR_TEAMLOGO       = GL.env("AE_TEAMLOGO_LAYER",      "TeamLogo");
  var LYR_TEAMC_SOLID    = GL.env("AE_TEAMCOLOR_SOLID",     "TeamColorSolid");

  var LEAGUE    = GL.env("AE_LEAGUE","NFL");
  var LIMIT_STR = GL.env("AE_LIMIT","");
  var LIMIT     = (LIMIT_STR && !isNaN(parseInt(LIMIT_STR,10))) ? parseInt(LIMIT_STR,10) : null;

  var OUTDIR    = GL.env("AE_OUTDIR","");
  var PATH_TPL  = GL.env("AE_PATH_TEMPLATE","{league}");
  var ANIM_NAME = GL.env("AE_ANIM","_LogoSpinText");
  var RS_TPL    = GL.env("AE_RS_TEMPLATE","Best Settings");
  var OM_TPL    = GL.env("AE_OM_TEMPLATE","PNG Sequence");
  var PURGE     = (GL.env("AE_PURGE_BEFORE_RENDER","1")==="1");
  var NO_RENDER = (GL.env("AE_NO_RENDER","0")==="1");
  var QUIT_APP  = (GL.env("AE_QUIT","1")==="1");

  // Logos dir/template
  var logoOpts = {
    dir:  GL.env("AE_LOGO_DIR",""),
    tpl:  GL.env("AE_LOGO_PATH_TEMPLATE","{league}/{abbr}"),
    exts: GL.env("AE_LOGO_EXTS","png,jpg,jpeg,svg,ai,psd")
  };

  // ---- Start ----
  if (app.beginSuppressDialogs){ try{ app.beginSuppressDialogs(); }catch(e){} }
  app.beginUndoGroup("Logo Spin Text — PNG Seq");

  if(!PROJECT) GL.fail("AE_PROJECT env not set.");
  var aep=new File(PROJECT); if(!aep.exists) GL.fail("AE_PROJECT not found: "+PROJECT);
  app.open(aep);

  if(!CSV_PATH) GL.fail("AE_CSV env not set.");
  if(!LEAGUE || GL.cleanValue(LEAGUE)==="") GL.fail("AE_LEAGUE is required.");

  // ---- CSV parse incl. team_id ----
  var rows = GL.parseCSV(GL.openRead(CSV_PATH));
  var H    = rows[0], M = {}; for (var i=0;i<H.length;i++) M[GL.toLower(H[i])] = i;
  function need(k){ if(M[k]===undefined) GL.fail("Missing column: "+k); }
  need("abbreviation"); need("r"); need("g"); need("b"); need("r2"); need("g2"); need("b2");
  function opt(){ for (var i=0;i<arguments.length;i++){ var k=arguments[i]; if (M[k]!==undefined) return M[k]; } return undefined; }

  var IDX = {
    league:  opt("league"),
    conf:    opt("conference","conf","group"),
    name:    opt("displayname","name"),
    team_id: opt("team_id")
  };

  function buildTeams(){
    var out=[], leagueDefault=LEAGUE;
    for (var r=1; r<rows.length; r++){
      var row = rows[r]; if (!row || !row.length) continue;
      var ab = GL.cleanValue(row[M["abbreviation"]]); if (!ab) continue;
      out.push({
        abbr: ab,
        league: (IDX.league!==undefined ? GL.cleanValue(row[IDX.league]) : GL.cleanValue(leagueDefault||"NA")),
        conference: (IDX.conf!==undefined ? GL.cleanValue(row[IDX.conf]) : ""),
        name: (IDX.name!==undefined ? GL.cleanValue(row[IDX.name]) : ab),
        team_id: (IDX.team_id!==undefined ? GL.cleanValue(row[IDX.team_id]) : ""),
        primary:   GL.rgb01(row[M["r"]],  row[M["g"]],  row[M["b"]]),
        secondary: GL.rgb01(row[M["r2"]], row[M["g2"]], row[M["b2"]])
      });
    }
    return out;
  }

  var teamsAll = buildTeams();
  var todo = (function(all, leagueStr){
    var res=[], targetBase = GL.normalizeLeagueBase(leagueStr);
    for (var i=0;i<all.length;i++){
      var teamBase = GL.normalizeLeagueBase(all[i].league);
      if (teamBase === targetBase) res.push(all[i]);
    }
    return res;
  })(teamsAll, LEAGUE);

  if (LIMIT && todo.length>LIMIT) todo = todo.slice(0, LIMIT);
  if(!todo.length) GL.fail("No teams matched league: "+LEAGUE);

  var comp = GL.findComp(COMP_NAME);
  if(!comp) GL.fail("Comp not found: "+COMP_NAME);

  var rootOut = OUTDIR ? new Folder(OUTDIR) : (app.project.file ? app.project.file.parent : Folder.desktop);
  GL.ensureFolder(rootOut);

  // ---- Logo import cache ----
  var __logoCache = {};
  function importTeamLogoFootage(team){
    var f = GL.findLogoFile(team.league, team.abbr, logoOpts);
    if (!f) return null;
    var key = f.fsName;
    if (__logoCache[key]) return __logoCache[key];
    try {
      var io = new ImportOptions(f);
      if (!io.canImportAs(ImportAsType.FOOTAGE)) return null;
      var footage = app.project.importFile(io);
      __logoCache[key] = footage;
      return footage;
    } catch(e){ return null; }
  }

  // ---- Comp-specific helpers ----
  function setTeamName(compItem, name, primary){
    var ly = GL.getLayer(compItem, LYR_TEAMNAME);
    if (!ly) return false;
    GL.setTextContent(ly, name);
    return GL.setTextOrFillColor(ly, primary); // stroke off
  }

  function setFieldGoal(compItem, secondary){
    var ly = GL.getLayerAlt(compItem, LYR_FIELDGOAL, ["FIELD GOAL"]);
    if (!ly) return false;
    return GL.setTextOrFillColor(ly, secondary); // stroke off
  }

  function setTeamColorSolid(compItem, primary){
    var ly = GL.getLayer(compItem, LYR_TEAMC_SOLID);
    if (!ly) return false;
    return GL.setSolidColor(ly, primary);
  }

  function replaceMainLogo(compItem, team){
    var ly = GL.getLayer(compItem, LYR_TEAMLOGO);
    if (!ly) return false;
    var footage = importTeamLogoFootage(team);
    if (!footage) return false;
    try{ ly.replaceSource(footage, false); return true; }catch(e){ return false; }
  }

  // Replace logo(s) inside a precomp layer (RotateLogo1/2)
  function replaceLogosInsidePrecompLayer(compItem, precompLayerName, team){
    var host = GL.getLayer(compItem, precompLayerName);
    if (!host || !host.source || !(host.source instanceof CompItem)) return 0;

    var inner   = host.source;
    var footage = importTeamLogoFootage(team);
    if (!footage) return 0;

    var changed = 0;
    var allowed = String(logoOpts.exts||"png,jpg,jpeg,svg,ai,psd").toLowerCase().split(",");
    function hasAllowedExt(file){
      try{
        if (!file || !file.exists) return false;
        var n = String(file.name||"").toLowerCase();
        for (var i=0;i<allowed.length;i++){
          var ext = "."+GL.trim(allowed[i].toLowerCase());
          if (n.slice(-ext.length)===ext) return true;
        }
      }catch(e){}
      return false;
    }

    for (var i=1;i<=inner.numLayers;i++){
      var L = inner.layer(i);
      if (!L || !L.source) continue;

      if (L.source instanceof FootageItem){
        var ms = L.source.mainSource;
        var good = false;
        if (L.source.file && hasAllowedExt(L.source.file)) good = true;
        if (!good && /teamlogo/i.test(String(L.name||""))) good = true;
        if (good && ms && ms.isStill === true){
          try{ L.replaceSource(footage, false); changed++; }catch(e){}
        }
      }
    }
    return changed;
  }

  // ================= Shatter color (robust) =================

  // darkened smart SECONDARY with 20/20/20 floor
  function darkerForShatter(base01){
    var factor = GL.numOr(GL.env("AE_DARKEN_FACTOR","0.5"), 0.5);
    var c = GL.safeColor(base01);
    var d = [c[0]*factor, c[1]*factor, c[2]*factor];
    var floor = 20/255.0;
    if (d[0] < floor && d[1] < floor && d[2] < floor) d = [floor, floor, floor];
    return d;
  }

  function _setColorAllKeys(prop, col){
    if (!prop || prop.propertyValueType !== PropertyValueType.COLOR) return 0;
    var changed = 0;
    try{
      var n = prop.numKeys || 0;
      if (n > 0){
        for (var k=1; k<=n; k++){
          var t = prop.keyTime(k);
          prop.setValueAtTime(t, col); changed++;
        }
      } else { prop.setValue(col); changed++; }
    }catch(e){}
    return changed;
  }

  function _setAllColorPropsDeep(effectGroup, col){
    var changed = 0;
    if (!effectGroup || effectGroup.numProperties === undefined) return 0;
    for (var i=1; i<=effectGroup.numProperties; i++){
      var p = effectGroup.property(i);
      if (!p) continue;
      try{
        if (p.propertyValueType === PropertyValueType.COLOR){
          changed += _setColorAllKeys(p, col);
        }else if (p.numProperties !== undefined && p.numProperties > 0){
          changed += _setAllColorPropsDeep(p, col);
        }
      }catch(e){}
    }
    return changed;
  }

  function _findShatterOnLayer(layer){
    var fx = layer && layer.property("Effects");
    if (!fx) return null;
    for (var i=1; i<=fx.numProperties; i++){
      var e  = fx.property(i);
      var mn = String(e && e.matchName || "");
      var nm = String(e && e.name      || "");
      if (mn === "ADBE Shatter" || mn === "Shatter" || /shatter/i.test(mn) || /shatter/i.test(nm)){
        return e;
      }
    }
    return null;
  }

  function _forceTextureModesToColor(shatter){
    var changed = 0;
    for (var i=1; i<=shatter.numProperties; i++){
      var p = shatter.property(i);
      if (!p) continue;
      var nm = String(p.name||"").toLowerCase();
      if (/front\s*mode|side\s*mode|back\s*mode/.test(nm)){
        try{
          var set=false;
          for (var idx=1; idx<=10; idx++){
            var s = p.getDisplayNameAtIndex ? String(p.getDisplayNameAtIndex(idx)).toLowerCase() : "";
            if (s && /color/.test(s)){ p.setValue(idx); set=true; changed++; break; }
          }
          if (!set){ try{ p.setValue(3); changed++; }catch(e){} }
        }catch(e){}
      }
      if (p.numProperties && p.numProperties>0) changed += _forceTextureModesToColor(p);
    }
    return changed;
  }

  // set on the layer; if not found, dive into its source precomp (depth-limited)
  function setShatterColorDeepOnLayer(compItem, layerName, color01, depth){
    depth = depth || 0;
    if (depth > 3) return 0;

    var ly = GL.getLayer(compItem, layerName);
    if (!ly) return 0;

    var total = 0, sh = _findShatterOnLayer(ly);
    if (sh){
      total += _setAllColorPropsDeep(sh, GL.safeColor(color01));
      if (GL.env("AE_SHATTER_FORCE_MODE","0")==="1") total += _forceTextureModesToColor(sh);
      return total;
    }

    try{
      if (ly.source && (ly.source instanceof CompItem)){
        var inner = ly.source;
        for (var i=1; i<=inner.numLayers; i++){
          var L = inner.layer(i);
          var sh2 = _findShatterOnLayer(L);
          if (sh2){
            total += _setAllColorPropsDeep(sh2, GL.safeColor(color01));
            if (GL.env("AE_SHATTER_FORCE_MODE","0")==="1") total += _forceTextureModesToColor(sh2);
          }
          try{
            if (L && L.source && (L.source instanceof CompItem) && depth < 3){
              total += setShatterColorDeepOnLayer(L.source, L.name, color01, depth+1);
            }
          }catch(e){}
        }
      }
    }catch(e){}

    return total;
  }

  // ---- Render Queue reset ----
  GL.rqClear();

  // ---- Loop teams ----
  for (var i=0;i<todo.length;i++){
    var t = todo[i];
    var smart = GL.computeSmartColors( GL.safeColor(t.primary), GL.safeColor(t.secondary) );
    var P = smart.primary, S = smart.secondary;

    // Text & color
    setTeamName(comp, t.abbr, P);
    setFieldGoal(comp, P);
    setTeamColorSolid(comp, P);

    // Logos
    replaceMainLogo(comp, t);
    replaceLogosInsidePrecompLayer(comp, LYR_ROTATELOGO1, t);
    replaceLogosInsidePrecompLayer(comp, LYR_ROTATELOGO2, t);

    // Shatter color on RotateLogo1/2 (layer or inside precomp)
    var shatterCol = darkerForShatter(S); // use smart SECONDARY, darkened & floored
    setShatterColorDeepOnLayer(comp, LYR_ROTATELOGO1, shatterCol, 0);
    setShatterColorDeepOnLayer(comp, LYR_ROTATELOGO2, shatterCol, 0);

    if (PURGE && app.purge){ try{ app.purge(PurgeTarget.ALL_CACHES); }catch(e){} }

    if (!NO_RENDER){
      var lc = GL.leagueAndConfForPath(t.league, t.conference);
      // CFB special: use team_id if present
      if (lc.base === "CFB" && t.team_id && String(t.team_id)!=="") lc.conf = String(t.team_id);

      var paths = GL.outPaths(new Folder(OUTDIR||Folder.desktop.fsName), PATH_TPL, lc.base, t.abbr, ANIM_NAME, lc.conf, t.espn_team_id);
      GL.rqRenderTo(comp, RS_TPL, OM_TPL, paths.file);
    }

    // Clean up logo from project after render
    var removed = GL.removeLogoByAbbr(t.abbr);
    if (removed > 0) $.writeln("Removed " + removed + " logo(s) for " + t.abbr);
  }

  app.endUndoGroup();
  if (app.endSuppressDialogs){ try{ app.endSuppressDialogs(); }catch(e){} }
  if (QUIT_APP) app.quit();
})();
